/******************************************************************************

 MRF24WB0M Driver Customization
 Module for Microchip TCP/IP Stack
  -Provides access to MRF24WB0M WiFi controller
  -Reference: MRF24WB0M Data sheet, IEEE 802.11 Standard

*******************************************************************************
 FileName:      WF_Config.c
 Dependencies:   TCP/IP Stack header files
 Processor:      PIC18, PIC24F, PIC24H, dsPIC30F, dsPIC33F, PIC32
 Compiler:      Microchip C32 v1.10b or higher
            Microchip C30 v3.22 or higher
            Microchip C18 v3.34 or higher
 Company:      Microchip Technology, Inc.

 Software License Agreement

 Copyright (C) 2002-2010 Microchip Technology Inc.  All rights reserved.

 Microchip licenses to you the right to use, modify, copy, and distribute:
 (i)  the Software when embedded on a Microchip microcontroller or digital 
      signal controller product ("Device") which is integrated into 
      Licensee's product; or
 (ii) ONLY the Software driver source files ENC28J60.c, ENC28J60.h,
      ENCX24J600.c and ENCX24J600.h ported to a non-Microchip device used in 
     conjunction with a Microchip ethernet controller for the sole purpose 
     of interfacing with the ethernet controller.

 You should refer to the license agreement accompanying this Software for 
 additional information regarding your rights and obligations.

 THE SOFTWARE AND DOCUMENTATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
 KIND, EITHER EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION, ANY WARRANTY
 OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, TITLE AND
 NON-INFRINGEMENT. IN NO EVENT SHALL MICROCHIP BE LIABLE FOR ANY INCIDENTAL,
 SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST
 OF PROCUREMENT OF SUBSTITUTE GOODS, TECHNOLOGY OR SERVICES, ANY CLAIMS BY
 THIRD PARTIES (INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), ANY CLAIMS
 FOR INDEMNITY OR CONTRIBUTION, OR OTHER SIMILAR COSTS, WHETHER ASSERTED ON
 THE BASIS OF CONTRACT, TORT (INCLUDING NEGLIGENCE), BREACH OF WARRANTY, OR
 OTHERWISE.


 Author            Date      Comment
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 KH                 27 Jan 2010 Created for MRF24WB0M
******************************************************************************/


#include "HardwareProfile.h"

#if defined(WF_CS_TRIS)

/*==========================================================================*/
/*                                  INCLUDES                                */
/*==========================================================================*/
#include "TCPIP Stack/TCPIP.h"

#if defined ( EZ_CONFIG_SCAN )
#include "TCPIP Stack/WFEasyConfig.h"
#endif /* EZ_CONFIG_SCAN */


/*==========================================================================*/
/*                                  DEFINES                                 */
/*==========================================================================*/

/* used for assertions */
#ifdef WF_DEBUG
    #define WF_MODULE_NUMBER    WF_MODULE_WF_CONFIG
#endif


/*****************************************************************************
 * FUNCTION: WF_ProcessEvent
*
 * RETURNS:  None
*
 * PARAMS:   event      -- event that occurred
 *           eventInfo  -- additional information about the event.  Not all events
 *                         have associated info, in which case this value will be
 *                         set to WF_NO_ADDITIONAL_INFO (0xff)
*
 *  NOTES:   The Host application must NOT directly call this function.  This 
 *           function is called by the WiFi Driver code when an event occurs
 *           that may need processing by the Host CPU.  
*
 *           No other WiFi Driver function should be called from this function, with the
*                   exception WF_ASSERT.  It is recommended that if the application wishes to be 
*                   notified of an event that it simply set a flag and let application code in the 
*                   main loop handle the event. 
*               
 *           WFSetFuncState must be called when entering and leaving this function.  
*                   When WF_ASSERT is enabled this allows a runtime check if any illegal WF functions 
*                   are called from within this function.
* 
 *           For events that the application is not interested in simply leave the
 *           case statement empty.
 *
 *
 *           Customize this function as needed for your application.
 *****************************************************************************/
 int1 g_WifiConnectFail = 0;
//int1 g_wifiConnected = 0;
//int1 g_WifiTempDisFlag = FALSE;
//TICK g_WifiTempDisTick;
//#define WF_SET_TEMP_DIS()  g_WifiTempDisFlag = TRUE; g_WifiTempDisTick = TickGet()
//#define WF_CLEAR_TEMP_DIS() g_WifiTempDisFlag = FALSE
void WF_ProcessEvent(UINT8 event, UINT16 eventInfo, UINT8 *extraInfo)
{
  #if defined(WF_DEBUG)
   tMgmtIndicateSoftAPEvent *softAPEvent;
  #endif
   
   //printf(SerialPutc, "WF_ProcessEvent(%X, %LX, %LX)\r\n", event, eventInfo, extraInfo);
   
    /* this function tells the WF driver that we are in this function */
    WFSetFuncState(WF_PROCESS_EVENT_FUNC, WF_ENTERING_FUNCTION);
      
    switch (event)
    {
        /*--------------------------------------*/
        case WF_EVENT_CONNECTION_SUCCESSFUL:
        /*--------------------------------------*/   
            wf_debug_printf(wf_debug_putc, "\r\nWIFI Event: Connection Successful\r\n");
            
            //g_wifiConnected = 1;
            //WF_CLEAR_TEMP_DIS();
            break;
        
        /*--------------------------------------*/            
        case WF_EVENT_CONNECTION_FAILED:
        /*--------------------------------------*/
            /* eventInfo will contain value from tWFConnectionFailureReasons */
            wf_debug_printf(wf_debug_putc, "\r\nWIFI Event: Connection Failed  -- eventInfo = %Lu", eventInfo);
            
            g_WifiConnectFail = 1;
            //g_wifiConnected = 0;
            //WF_CLEAR_TEMP_DIS();            
            break; 
            
        /*--------------------------------------*/
        case WF_EVENT_CONNECTION_TEMPORARILY_LOST:
        /*--------------------------------------*/
            wf_debug_printf(wf_debug_putc, "\r\nWIFI Event: Connection Temporarily Lost -- eventInfo = %Lu", eventInfo);
            #if defined(STACK_USE_CCS_SCAN_TASK)
            WIFIScanIgnoreResults();
            #endif
            //g_wifiConnected = 0;
            //WF_SET_TEMP_DIS();
            break;
            
        /*--------------------------------------*/
        case WF_EVENT_CONNECTION_PERMANENTLY_LOST:            
        /*--------------------------------------*/
            wf_debug_printf(wf_debug_putc, "\r\nWIFI Event: Connection Permanently Lost -- eventInfo =  %Lu", eventInfo);
            g_WifiConnectFail = 1;
            //g_wifiConnected = 0;
            //WF_CLEAR_TEMP_DIS();
            break;
            
        /*--------------------------------------*/    
        case WF_EVENT_CONNECTION_REESTABLISHED:
        /*--------------------------------------*/
            wf_debug_printf(wf_debug_putc, "\r\nWIFI Event: Connection Reestablished\r\n");
            #if defined(STACK_USE_CCS_SCAN_TASK)
            WIFIScanIgnoreResults();
            #endif                
            //g_wifiConnected = 1;
            //WF_CLEAR_TEMP_DIS();
            break;
            
        /*--------------------------------------*/    
        case WF_EVENT_SCAN_RESULTS_READY:
        /*--------------------------------------*/  
            wf_debug_printf(wf_debug_putc, "\r\nWIFI Event: Scan Results Ready,%Lu results\r\n", eventInfo);
            #if defined ( EZ_CONFIG_SCAN ) && !defined(__18CXX)
            WFScanEventHandler(eventInfo);
            #endif /* EZ_CONFIG_SCAN */
            #if defined(STACK_USE_CCS_SCAN_TASK)
            WIFIScanResults(eventInfo);
            #endif
            break;

        case WF_EVENT_KEY_CALCULATION_REQUEST:
            #if defined(DERIVE_KEY_FROM_PASSPHRASE_IN_HOST)
                g_WpsPassphrase.valid = TRUE;
                memcpy((void *)&g_WpsPassphrase.passphrase, (void *)extraInfo, sizeof(g_WpsPassphrase.passphrase));
                #if defined(STACK_USE_UART)
                    putrsUART("Event: WPS pass phrase Ready\r\n");
                #endif /* STACK_USE_UART */
            #endif  /* defined(DERIVE_KEY_FROM_PASSPHRASE_IN_HOST) */    
            break;  

        /*--------------------------------------*/                            
        case WF_EVENT_RX_PACKET_RECEIVED:
        /*--------------------------------------*/   
            break;         

    case WF_EVENT_SOFT_AP_EVENT:
#if defined(WF_DEBUG)
    {
        softAPEvent = (tMgmtIndicateSoftAPEvent *) extraInfo;
        uint8_t *addr = softAPEvent->address;
        wf_debug_printf(wf_debug_putc, "Event: SoftAP, ");
        if (softAPEvent->event == SOFTAP_EVENT_CONNECTED) 
        {
            wf_debug_printf(wf_debug_putc, "Connected ");
        } 
        else if (softAPEvent->event == SOFTAP_EVENT_DISCONNECTED) 
        {
            wf_debug_printf(wf_debug_putc, "Disconnected ");
            if (softAPEvent->reason == SOFTAP_EVENT_LINK_LOST)
            {
                wf_debug_printf(wf_debug_putc, "LinkLost ");
            }
            else if (softAPEvent->reason == SOFTAP_EVENT_RECEIVED_DEAUTH)
            {
                wf_debug_printf(wf_debug_putc, "ReceivedDeauth ");
            }
        }
        wf_debug_printf(wf_debug_putc, "%x:%x:%x:%x:%x:%x\r\n", addr[0], addr[1], addr[2], addr[3], addr[4], addr[5]);
    }
#endif /* STACK_USE_UART */
        break;

        default:
            WF_ASSERT(FALSE);  /* unknown event */
            break;
    }        
    
    /* Informs the WF driver that we are leaving this function */
    WFSetFuncState(WF_PROCESS_EVENT_FUNC, WF_LEAVING_FUNCTION);
}    
  
 
/*
*********************************************************************************************************
*                                   WF_AssertionFailed()
*    
* Description : Called by a WiFi library function when an assert occurs.
*
* Argument(s) : moduleNumber - module number (located in WFApi.h)
*                
*               lineNumber   - line number within module where assert occurred.
*
* Return(s)   : None
*
* Caller(s)   : WF Driver
*
* Notes:      : (1) If the WF_ASSERT macro is enabled (via the WF_DEBUG define in WF_Config.h) this is the 
*                   function that gets called when WF_ASSERT() fails.
*         
*               (2) Customize this function as desired to handle assertion errors
*
*********************************************************************************************************
*/
#define WIFI_ASSERT_STRING "WiFi Assert     M:"
#if !defined(__WF_OVERRIDE_ASSERTION_DISPLAY__)
void WF_AssertionFailed(UINT8 moduleNumber, UINT16 lineNumber) 
{
    wf_debug_printf(wf_debug_putc, "\r\nWF ASSERTION: Module Number = ");
    
    wf_debug_printf(wf_debug_putc, "%u  ", moduleNumber);
    
    wf_debug_printf(wf_debug_putc, "Line Number = ");
    
    wf_debug_printf(wf_debug_putc, "%Lu", lineNumber);
    
    for(;;);
}
#endif
#endif /* WF_CS_TRIS */

